<?php

if (!class_exists('WP_List_Table')) {
	require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );
}

/**
 * Shows Entries table in the Entries tab on the Edit Content Upgrade page
 */
class AopsEntriesTab extends WP_List_Table {

	private $cu_id = 0;

	function __construct($cu_id = 0) {

		parent::__construct(array(
			'singular' => __('entry', 'aops'), //singular name of the listed records
			'plural' => __('entries', 'aops'), //plural name of the listed records
			'ajax' => false  //does this table support ajax?
		));

		$this->cu_id = $cu_id;
	}

	/**
	 * Decide which columns to activate the sorting functionality on
	 * @return array $sortable, the array of columns that can be sorted by the user
	 */
	public function get_sortable_columns() {
		$sortable = array(
			'col_cu_id'	=> array('cu_id', false),
			'col_name'	=> array('name', false),
			'col_email'	=> array('email', false),
			'col_created_at' => array('created_at', false),
		);
		return $sortable;
	}

	/**
	 * Define the columns that are going to be used in the table
	 * @return array $columns, the array of columns to use with the table
	 */
	function get_columns() {
		$columns = array(
			'col_name'			=> __('First name'),
			'col_email'			=> __('Email'),
			'col_created_at'	=> __('Submitted'),
			'col_actions'		=> __('Actions')
		);
		return $columns;
	}

	/**
	* Prepare the table with different parameters, pagination, columns and table elements
	*/
	function prepare_items() {
		global $wpdb, $_wp_column_headers;
		$screen = get_current_screen();

		if ($this->cu_id) {
			$where = " cu_id = " . $this->cu_id;
		}
		else {
			$where = " 1=1 ";
		}

		/* -- Process delete action -- */
		$this->process_action();


		$tableName = $wpdb->prefix . 'cu_entries';
		$query = "SELECT * FROM " . $tableName . ' WHERE ' . $where;

		/* -- Ordering parameters -- */

		//Parameters that are going to be used to order the result
		$orderby = !empty($_GET["orderby"]) ? htmlentities($_GET["orderby"], ENT_QUOTES) : 'created_at';
		$order = !empty($_GET["order"]) ? htmlentities($_GET["order"], ENT_QUOTES) : 'DESC';

		if (!empty($orderby) & !empty($order)) {
			$query .= $wpdb->prepare( ' ORDER BY "%s" "%s" ', $orderby, $order );
		}

		/* -- Pagination parameters -- */

		//Number of elements in your table?
		$countQuery = "SELECT count(*) FROM " . $tableName. ' WHERE ' . $where;
		$totalItems = $wpdb->get_var($countQuery); //return the total number of affected rows

		//How many to display per page?
		$user = get_current_user_id();
		$option = $screen->get_option('per_page', 'option');

		$perPage = 30; //get_user_meta($user, $option, true);

		if ( empty ( $perPage) || $perPage < 1 ) {

			$perPage = $screen->get_option( 'per_page', 'default' );

		}

		//Which page is this?
		$paged = !empty($_GET["paged"]) ? intval($_GET["paged"]) : '';

		//Page Number
		if ( empty($paged) || !is_numeric($paged) || $paged<=0 ) { $paged = 1; }

		//How many pages do we have in total?
		$totalPages = ceil($totalItems/$perPage);

		//adjust the query to take pagination into account
		if(!empty($paged) && !empty($perPage)){
			$offset=($paged-1) * $perPage;
			$query.=' LIMIT ' . (int)$offset . ',' . (int)$perPage;
		}

		/* -- Register the pagination -- */
		$this->set_pagination_args( array(
		   "total_items" => $totalItems,
		   "total_pages" => $totalPages,
		   "per_page" => $perPage,
		) );

		//The pagination links are automatically built according to those parameters

		/* -- Register the Columns -- */
		$columns = $this->get_columns();
		$hidden = array();
		$sortable = $this->get_sortable_columns();

		$this->_column_headers = array($columns, $hidden, $sortable);

		/* -- Fetch the items -- */
		$this->items = $wpdb->get_results($query);
	}


	public function process_action() {

		// Detect when a bulk action is being triggered...
		if ( isset( $_REQUEST['entry_action'] ) && 'delete_entry' === $_REQUEST['entry_action'] ) {

			// In our file that handles the request, verify the nonce.
			$nonce = esc_attr( $_REQUEST['_wpnonce'] );

			if ( ! wp_verify_nonce( $nonce, 'cu_delete_entry' ) ) {
			  echo( '<br>wp_nonce is invalid, skipping.' );
			}
			else {
			  self::delete_entry( absint( $_GET['entry_id'] ) );
			}
		}
	}

	static function delete_entry($id) {
		global $wpdb;
		$tableName = $wpdb->prefix . 'cu_entries';
		$query = $wpdb->prepare( "DELETE FROM " . $tableName . " WHERE id = %d ", $id );
		$wpdb->query($query);
	}

	/**
	 * Display the rows of records in the table
	 * @return string, echo the markup of the rows
	 */
	function display_rows() {

		//Get the records registered in the prepare_items method
		$records = $this->items;

		//Get the columns registered in the get_columns and get_sortable_columns methods
		list( $columns, $hidden ) = $this->get_column_info();

		//Loop for each record
		if (!empty($records)) {
			foreach ($records as $rec) {

				//Open the line
				echo '<tr id="record_'.$rec->id.'">';

				foreach ( $columns as $column_name => $column_display_name ) {

					//Style attributes for each col
					$class = "class='$column_name column-$column_name'";
					$style = "";
					if ( in_array( $column_name, $hidden ) ) $style = ' style="display:none;"';
					$attributes = $class . $style;

					//Display the cell
					switch ( $column_name ) {
						case "col_name": echo '<td '.$attributes.'>' . esc_html($rec->name) . '</td>'; break;
						case "col_email": echo '<td '.$attributes.'>' . esc_html($rec->email) . '</td>'; break;
						case "col_cu_id":
							$contentUpgradeTitle = get_the_title($rec->cu_id);
							$href = '/wp-admin/post.php?post=' . $rec->cu_id . '&action=edit';
							echo '<td '.$attributes.'><a href="' . $href . '">' . $contentUpgradeTitle . '</a></td>';
							break;
						case "col_created_at": echo '<td '.$attributes . '>' . $rec->created_at . '</td>'; break;
						case "col_actions":
							// create a nonce
							$delete_nonce = wp_create_nonce( 'cu_delete_entry' );
							$link = sprintf( '<a href="post.php?post=' . $this->cu_id  . '&action=edit&entry_action=%s&entry_id=%s&_wpnonce=%s#cu_tab_entries">Delete</a>', 'delete_entry', absint( $rec->id ), $delete_nonce );
							echo "<td>$link</td>";
							break;
					}
				}

				//Close the line
				echo'</tr>';
			}
		}
	}

}

?>
